import contextlib
import subprocess
from abc import ABC, abstractmethod
from pathlib import Path

import text
from aui import InfoDialogWindow, ProgressbarDialogWindow


class Converter(ABC):
    """
    A base abstract class for file converters.

    Attributes:
        command (list): The command to execute for the conversion process.
        file (Path): The input file to be converted.
        format (str): The format to convert the file to.
        target_file (Path): The output file after conversion.
        timeout_ms (int): The timeout in milliseconds for the conversion process.
        _process (subprocess.Popen): The subprocess object for the conversion process.

    Methods:
        build_command(self) -> None: Abstract method to build the command for the conversion process.
        convert(self) -> bool: Initiates the conversion process and displays a progress bar.
        valid_target_file(self) -> None: Checks and generates a valid target file name.
        _error_dialog(self) -> None: Displays an error dialog if an error occurs during conversion.
        _delete_target_file(self) -> None: Deletes the target file if it exists.
        _handle_cancel(self) -> bool: Handles the cancellation of the conversion process.
        _handle_progress(self, _, window: ProgressbarDialogWindow) -> bool: Handles the progress of the conversion process.
        _handle_process(self) -> bool: Handles the conversion process.
        _success_dialog(self) -> None: Displays a success dialog after successful conversion.

    This class should be subclassed to implement the 'build_command' method for specific file conversion logic.
    """

    def __init__(self, file: Path, format: str, **kwargs):
        self.command: list = []
        self.file: Path = file
        self.format: str = format
        self.target_file: Path = self.file.with_suffix(f".{self.format.lower()}")
        self.timeout_ms: int = kwargs.get("timeout_ms", 60)

        self._process: subprocess.Popen = None
        self.valid_target_file()
        self.build_command()

    @abstractmethod
    def build_command(self) -> None:
        """
        Abstract method to build the command for the conversion process.

        This method should be implemented by subclasses to define
        the specific command required for converting the input file to the desired format.
        """
        pass

    def valid_target_file(self) -> None:
        """
        Checks and generates a valid target file name.

        The method first sets the suffix of the target file based on the format attribute converted to lowercase.
        Then, it iterates over possible target file names until it finds a name that does not exist in the filesystem.
        The target file name is generated by appending a count number in parentheses to the stem of the input file name.

        This method ensures that the target file name is unique and does not overwrite any existing files.
        """
        suffix = self.format.lower()
        count = 1

        while self.target_file.exists():
            self.target_file = self.file.with_name(
                f"{self.file.stem} ({count}).{suffix}"
            )
            count += 1

    def convert(self) -> bool:
        """
        Initiates the conversion process by executing the command stored in 'self.command' using subprocess.Popen.
        Handles the conversion process by calling the '_handle_process' method, which displays a progress bar and manages the conversion progress.
        In case of any exceptions during the conversion process, displays an error dialog using the '_error_dialog' method.
        Returns True if the conversion process is successful, False otherwise.
        """
        if not self.command:
            return False

        try:
            with subprocess.Popen(
                self.command,
                stderr=subprocess.PIPE,
                stdout=subprocess.DEVNULL,
            ) as self._process:
                return self._handle_process()

        except Exception:
            self._error_dialog()
            return False

    def _handle_process(self):
        """
        Handles the conversion process.

        This method creates a ProgressbarDialogWindow to display the progress of the conversion process.
        It checks for cancellation of the process and error conditions.
        If the process completes successfully, it displays a success dialog; otherwise, it shows an error dialog.
        Returns True if the conversion process is successful, False otherwise.
        """
        window = ProgressbarDialogWindow(
            title=text.CONVERTING_TITLE,
            message=text.CONVERTING_LABEL.format(format=self.format),
            timeout_callback=self._handle_progress,
            timeout_ms=self.timeout_ms,
        )

        window.run()
        window.destroy()

        if self._handle_cancel():
            return False

        if self._process.returncode != 0:
            self._error_dialog()
            return False

        self._success_dialog()
        return True

    def _handle_cancel(self) -> bool:
        """
        Deletes the target file if the conversion process is canceled.

        If the conversion process is canceled (i.e., the process has not completed and the return code is None),
        this method terminates the process and deletes the target file using the '_delete_target_file' method.
        This ensures that any partially converted or erroneous target file is removed when the conversion process is canceled.
        Returns True if the process is canceled and the target file is deleted; otherwise, returns False.
        """
        if self._process.poll() != 0 and self._process.returncode is None:
            self._process.terminate()
            self._delete_target_file()
            return True
        return False

    def _delete_target_file(self) -> None:
        """
        Deletes the target file if it exists.

        If the target file exists in the filesystem, this method will delete it using the 'unlink' method from the Path class.
        This ensures that any existing target file is removed before a new conversion process starts.
        """
        if self.target_file.exists():
            self.target_file.unlink()

    def _success_dialog(self) -> None:
        """
        Displays a success dialog after a successful conversion.

        This method creates an InfoDialogWindow with the title set to the SUCCESS_TITLE constant
        and the message formatted with the converted format.
        The InfoDialogWindow is then run to display the success message to the user.
        """
        InfoDialogWindow(
            title=text.SUCCESS_TITLE,
            message=text.SUCCESS_MESSAGE.format(format=self.format),
        ).run()

    def _error_dialog(self) -> None:
        """
        Displays an error dialog after an error occurs during the conversion process.

        This method creates an InfoDialogWindow with the title set to the ERROR_TITLE constant
        and the message formatted with the ERROR_MESSAGE constant.
        The InfoDialogWindow is then run to display the error message to the user.
        """
        InfoDialogWindow(
            title=text.ERROR_TITLE,
            message=text.ERROR_MESSAGE,
        ).run()

    def _handle_progress(self, _, window: ProgressbarDialogWindow) -> bool:
        """
        Handles the progress of the conversion process.

        This method is responsible for updating the progress bar window during the conversion process.
        It checks if the conversion process is still running and updates the progress bar accordingly.
        If the conversion process has completed, it terminates the process and stops the progress bar.
        Returns True if the conversion process is still ongoing, and False if the process has completed.
        """
        if self._process and self._process.poll() is None:
            with contextlib.suppress(Exception):
                window.progressbar.pulse()

        if self._process.poll() is not None:
            self._process.terminate()
            window.stop()
            window.destroy()
            return False

        return True


class ImageConverter(Converter):
    """
    A subclass of Converter that implements the 'build_command' method for image file conversion.

    Attributes:
        Inherits all attributes from the base Converter class:
            - file (Path): The input image file to be converted.
            - format (str): The format to convert the image file to.
            - target_file (Path): The output image file after conversion.

    Methods:
        Inherits all methods from the base Converter class:
            - build_command(self) -> None: Abstract method to build the command for the conversion process.
            - valid_target_file(self) -> None: Checks and generates a valid target file name.
            - convert(self) -> bool: Initiates the conversion process and displays a progress bar.
            - _delete_target_file(self) -> None: Deletes the target file if it exists.
            - _success_dialog(self) -> None: Displays a success dialog after successful conversion.
            - _error_dialog(self) -> None: Displays an error dialog if an error occurs during conversion.
            - _handle_progress(self, _, window: ProgressbarDialogWindow) -> bool: Handles the progress of the conversion process.

    This class should be instantiated with an image file and the desired format for conversion.
    The 'build_command' method is implemented to define the specific command required for converting the input image file to the desired format.
    """

    def build_command(self) -> None:
        self.command = [
            "convert",
            "-auto-orient",
            str(self.file),
            str(self.target_file),
        ]


class VideoConverter(Converter):
    """
    A class for converting video files using FFmpeg.

    This class inherits from the base abstract class 'Converter' and implements the 'build_command' method specific to video file conversion using FFmpeg.

    Attributes:
        Inherits attributes from the 'Converter' class:
            file (Path): The input video file to be converted.
            format (str): The format to convert the video file to.
            target_file (Path): The output video file after conversion.

    Methods:
        Inherits methods from the 'Converter' class:
            build_command(self) -> None: Method to build the FFmpeg command for video conversion.

    This class should be used to convert video files to different formats by implementing the 'build_command' method with the appropriate FFmpeg command.
    """

    def build_command(self) -> None:
        self.command = [
            "ffmpeg",
            "-i",
            str(self.file),
            str(self.target_file),
        ]


class AudioConverter(Converter):
    """
    A subclass of Converter that implements the 'build_command' method specifically for audio file conversion using FFmpeg.

    Attributes:
        Inherits all attributes from the base Converter class:
            file (Path): The input audio file to be converted.
            format (str): The audio format to convert the file to.
            target_file (Path): The output audio file after conversion.

    Methods:
        build_command(self) -> None: Builds the FFmpeg command for audio file conversion.

    This class provides the necessary implementation for audio file conversion using FFmpeg by defining the 'build_command' method.
    """

    def build_command(self) -> None:
        self.command = [
            "ffmpeg",
            "-i",
            str(self.file),
            str(self.target_file),
        ]
