const Desklet = imports.ui.desklet;
const St = imports.gi.St;
const GLib = imports.gi.GLib;
const Util = imports.misc.util;
const Mainloop = imports.mainloop;
const Lang = imports.lang;
const Settings = imports.ui.settings;
const Main = imports.ui.main;
const Clutter = imports.gi.Clutter;
const Cairo = imports.cairo;
const Gio = imports.gi.Gio;
const Gettext = imports.gettext;
const Pango = imports.gi.Pango;

const UUID = "diskspace@schorschii";
const DESKLET_ROOT = imports.ui.deskletManager.deskletMeta[UUID].path;

// translation support
function _(str) {
	return Gettext.dgettext(UUID, str);
}

function MyDesklet(metadata, desklet_id) {
	// translation init: if installed in user context, switch to translations in user's home dir
	if(!DESKLET_ROOT.startsWith("/usr/share/")) {
		Gettext.bindtextdomain(UUID, GLib.get_home_dir() + "/.local/share/locale");
	}
	this._init(metadata, desklet_id);
}

function main(metadata, desklet_id) {
	return new MyDesklet(metadata, desklet_id);
}


MyDesklet.prototype = {
	__proto__: Desklet.Desklet.prototype,

	_init: function(metadata, desklet_id) {
		Desklet.Desklet.prototype._init.call(this, metadata);

		// initialize settings
		this.settings = new Settings.DeskletSettings(this, this.metadata["uuid"], desklet_id);
		this.settings.bindProperty(Settings.BindingDirection.IN, "size-prefix", "size_prefix", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "size-fraction-digits", "size_fraction_digits", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "reserved-blocks-as-used-space", "reserved_blocks_as_used_space", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "hide-decorations", "hide_decorations", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "use-custom-label", "use_custom_label", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "custom-label", "custom_label", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "scale-size", "scale_size", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "font", "font_raw", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "font-sub", "font_sub_raw", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "text-color", "text_color", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "design", "design", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "draw-free-space", "draw_free_space", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "fill-circle-background", "fill_circle_background", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "fill-circle-background-color", "fill_circle_background_color", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "circle-color", "circle_color", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "circle-color-free-space", "circle_color_free_space", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "use-own-circle-color", "use_own_circle_color", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "filesystem", "filesystem", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "type", "type", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "text-view", "text_view", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.IN, "onclick-action", "onclick_action", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.BIDIRECTIONAL, "random-circle-color-generated", "random_circle_color_generated", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.BIDIRECTIONAL, "random-circle-color-r", "random_circle_color_r", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.BIDIRECTIONAL, "random-circle-color-g", "random_circle_color_g", this.on_setting_changed);
		this.settings.bindProperty(Settings.BindingDirection.BIDIRECTIONAL, "random-circle-color-b", "trandom_circle_color_b", this.on_setting_changed);
		if(!this.random_circle_color_generated) {
			this.random_circle_color_r = Math.random();
			this.random_circle_color_g = Math.random();
			this.random_circle_color_b = Math.random();
			this.random_circle_color_generated = true;
		}

		// fallback to "/" if filesystem is not defined in the settings
		if (this.filesystem === undefined || this.filesystem === null) { 
			this.filesystem = "/"
		}

		// initialize desklet gui
		this.setupUI();
	},

	setupUI: function() {
		// defaults and initial values
		this.default_size = 150;

		// create and set root element
		this.canvas = new Clutter.Actor();
		this.textpercent = new St.Label({style_class:"textpercent"});
		this.textsub = new St.Label({style_class:"textsub"});
		this.textsub2 = new St.Label({style_class:"textsub2"});
		this.canvas.remove_all_children();
		this.canvas.add_actor(this.textpercent);
		this.canvas.add_actor(this.textsub);
		this.canvas.add_actor(this.textsub2);
		this.setContent(this.canvas);

		// set decoration settings
		this.refreshDecoration();

		// set initial values
		this.update();
	},

	update: function() {
		// set object sizes without recalc
		this.refreshDesklet();
		//global.log("update "+ this.filesystem); // debug

		this.font = this.parseFontStringToCSS(this.font_raw);
		this.fontSub = this.parseFontStringToCSS(this.font_sub_raw);
		this.default_size_font = this.font["font-size"];
		this.default_size_font_sub = this.fontSub["font-size"];


		// refresh again in two seconds
		this.timeout = Mainloop.timeout_add_seconds(5, Lang.bind(this, this.update));
	},

	refreshDesklet: function() {
		// get filesystem
		var type = this.type;
		var fs = decodeURIComponent(this.filesystem.replace("file://", "").trim());
		if(fs == null || fs == "") fs = "/";

		var percentString = "---";
		var avail = 0;
		var use = 0;
		var size = 0;

		// get values from command
		if(type == "ram" || type == "swap") {

			let file = Gio.file_new_for_path("/proc/meminfo");
			file.load_contents_async(null, (file, response) => {
				try {
					let [success, contents, tag] = file.load_contents_finish(response);
					if(success) {
						let mem = contents.toString();
						if(type == "ram") {
							size = parseInt(mem.match(/(MemTotal):\D+(\d+)/)[2]) * 1024;
							use = size - parseInt(mem.match(/(MemAvailable):\D+(\d+)/)[2]) * 1024;
						} else if(type == "swap") {
							size = parseInt(mem.match(/(SwapTotal):\D+(\d+)/)[2]) * 1024;
							use = size - parseInt(mem.match(/(SwapFree):\D+(\d+)/)[2]) * 1024;
						}
						avail = size - use;
						if(size > 0) {
							percentString = Math.round(use * 100 / size) + "%";
						}
						this.redraw(type, fs, avail, use, size, percentString);
						//global.log("avail:"+avail+" used:"+use); // debug
					}
				} catch(ex) {
					global.log("error getting RAM info: "+ex.toString());
				}
			});

		} else {

			// https://docs.gtk.org/gio/vfunc.File.query_filesystem_info_async.html
			let file = Gio.file_new_for_path(fs);
			file.query_filesystem_info_async(
				Gio.FILE_ATTRIBUTE_FILESYSTEM_USED
				+","+Gio.FILE_ATTRIBUTE_FILESYSTEM_FREE
				+","+Gio.FILE_ATTRIBUTE_FILESYSTEM_SIZE,
				1, null, (source_object, response, data) => {
					try {
						let fileInfo = file.query_filesystem_info_finish(response);
						avail = fileInfo.get_attribute_uint64(Gio.FILE_ATTRIBUTE_FILESYSTEM_FREE);
						size = fileInfo.get_attribute_uint64(Gio.FILE_ATTRIBUTE_FILESYSTEM_SIZE);
						if(this.reserved_blocks_as_used_space) {
							use = size - avail;
						} else {
							use = fileInfo.get_attribute_uint64(Gio.FILE_ATTRIBUTE_FILESYSTEM_USED);
						}
						percentString = Math.round(use * 100 / size) + "%";
					} catch(err) {
						// e.g. file not found (= not mounted)
						//global.log("error getting filesystem info: "+fs);
					}
					this.redraw(type, fs, avail, use, size, percentString);
				}
			);

		}
	},

	redraw: function(type, fs, avail, use, size, percentString) {
		// calc new sizes based on scale factor
		let absoluteSize = this.default_size * this.scale_size;
		let fontSize = Math.round(this.default_size_font * this.scale_size);
		let fontSizeSub = Math.round(this.default_size_font_sub * this.scale_size);
		let design = this.design;
		let drawFreeSpace = this.draw_free_space;

		// determine colors
		var circle_r = 1;
		var circle_g = 1;
		var circle_b = 1;
		var circle_a = 1;
		if(this.use_own_circle_color) {
			let circle_colors = this.circle_color.match(/\((.*?)\)/)[1].split(","); // get contents inside brackets: "rgb(...)"
			circle_r = parseInt(circle_colors[0])/255;
			circle_g = parseInt(circle_colors[1])/255;
			circle_b = parseInt(circle_colors[2])/255;
			if(circle_colors.length >= 4) circle_a = parseFloat(circle_colors[3]);
		} else {
			circle_r = this.random_circle_color_r;
			circle_g = this.random_circle_color_g;
			circle_b = this.random_circle_color_b;
		}

		// canvas setup
		let canvas = new Clutter.Canvas();
		canvas.set_size(absoluteSize * global.ui_scale, absoluteSize * global.ui_scale);
		canvas.connect("draw", (canvas, cr, width, height) => {
			cr.save();
			cr.setOperator(Cairo.Operator.CLEAR);
			cr.paint();
			cr.restore();
			cr.setOperator(Cairo.Operator.OVER);
			cr.scale(width, height);
			cr.translate(0.5, 0.5);

			let offset = Math.PI*0.5;
			let start = 0 - offset;
			let end = ((use*(Math.PI*2))/size) - offset;

			const free_space_colors = this.circle_color_free_space.match(/\((.*?)\)/)[1].split(","); // get contents inside brackets: "rgb(...)"
			const free_space_color_r = this.use_own_circle_color ? parseInt(free_space_colors[0])/255 : 1;
			const free_space_color_g  = this.use_own_circle_color ? parseInt(free_space_colors[1])/255 : 1;
			const free_space_color_b  = this.use_own_circle_color ? parseInt(free_space_colors[2])/255 : 1;
			const free_space_color_a =  this.use_own_circle_color ? (free_space_colors.length >= 4 ? parseFloat(free_space_colors[3]) : 1.0) : 0.2;

			if(design == "thin") {
				if(this.fill_circle_background) {
					const bg_fill_colors = this.fill_circle_background_color.match(/\((.*?)\)/)[1].split(",");
					const bg_fill_r = parseInt(bg_fill_colors[0])/255;
					const bg_fill_g = parseInt(bg_fill_colors[1])/255;
					const bg_fill_b = parseInt(bg_fill_colors[2])/255;
					const bg_fill_a = bg_fill_colors.length >= 4 ? parseFloat(bg_fill_colors[3]) : 1.0;
					cr.setSourceRGBA(bg_fill_r, bg_fill_g, bg_fill_b, bg_fill_a);
					cr.arc(0, 0, 0.45 - (0.045 / 2), 0, Math.PI*2);
					cr.fill();
				}
				if(drawFreeSpace) {
					cr.setSourceRGBA(free_space_color_r, free_space_color_g, free_space_color_b, free_space_color_a);
					cr.setLineWidth(0.045);
					cr.arc(0, 0, 0.45, 0, Math.PI*2);
					cr.stroke();
				}
				if(size > 0) {
					cr.setLineCap(Cairo.LineCap.ROUND);
					cr.setSourceRGBA(circle_r, circle_g, circle_b, circle_a);
					cr.setLineWidth(0.045);
					cr.arc(0, 0, 0.45, start, end);
					cr.stroke();
				}
			} else if(design == "compact") {
				if(drawFreeSpace) {
					cr.setSourceRGBA(free_space_color_r, free_space_color_g, free_space_color_b, free_space_color_a);
					cr.setLineWidth(0.4);
					cr.arc(0, 0, 0.2, 0, Math.PI*2);
					cr.stroke();
				}
				if(size > 0) {
					cr.setSourceRGBA(circle_r, circle_g, circle_b, circle_a);
					cr.setLineWidth(0.4);
					cr.arc(0, 0, 0.2, start, end);
					cr.stroke();
				}
			} else { // classic design
				if(this.fill_circle_background) {
					const bg_fill_colors = this.fill_circle_background_color.match(/\((.*?)\)/)[1].split(",");
					const bg_fill_r = parseInt(bg_fill_colors[0])/255;
					const bg_fill_g = parseInt(bg_fill_colors[1])/255;
					const bg_fill_b = parseInt(bg_fill_colors[2])/255;
					const bg_fill_a = bg_fill_colors.length >= 4 ? parseFloat(bg_fill_colors[3]) : 1.0;
					cr.setSourceRGBA(bg_fill_r, bg_fill_g, bg_fill_b, bg_fill_a);
					cr.arc(0, 0, 0.4 - (0.19 / 2), 0, Math.PI*2);
					cr.fill();
				}
				if(drawFreeSpace) {
					cr.setSourceRGBA(free_space_color_r, free_space_color_g, free_space_color_b, free_space_color_a);
					cr.setLineWidth(0.19);
					cr.arc(0, 0, 0.4, 0, Math.PI*2);
					cr.stroke();
				}
				if(size > 0) {
					cr.setSourceRGBA(circle_r, circle_g, circle_b, circle_a);
					cr.setLineWidth(0.19);
					cr.arc(0, 0, 0.4, start, end);
					cr.stroke();
					/////
					cr.setSourceRGBA(0, 0, 0, 0.1446);
					cr.setLineWidth(0.048);
					cr.arc(0, 0, 0.329, start, end);
					cr.stroke();
				}
				fontSize -= 3;
				fontSizeSub -= 3;
			}

			return true;
		});
		canvas.invalidate();
		this.canvas.set_content(canvas);
		this.canvas.set_size(absoluteSize * global.ui_scale, absoluteSize * global.ui_scale);

		let textSub1 = "";
		let textSub2 = "";
		let name = "";
		if(type == "ram") {
			name = this.shortText(_("RAM"));
		} else if(type == "swap") {
			name = this.shortText(_("Swap"));
		} else if(fs == "/") {
			name = this.shortText(_("Filesystem"));
		} else {
			let pathparts = fs.split("/");
			name = this.shortText(pathparts[pathparts.length-1]);
		}
		if(this.text_view == "name-size") {
			textSub1 = name;
			textSub2 = this.niceSize(size);
		} else if(this.text_view == "name-free") {
			textSub1 = name;
			textSub2 = this.niceSize(avail);
		} else if(this.text_view == "used-size") {
			textSub1 = this.niceSize(use);
			textSub2 = this.niceSize(size);
		} else if(this.text_view == "size-used") {
			textSub1 = this.niceSize(size);
			textSub2 = this.niceSize(use);
		} else if(this.text_view == "free-size") {
			textSub1 = this.niceSize(avail);
			textSub2 = this.niceSize(size);
		} else if(this.text_view == "size-free") {
			textSub1 = this.niceSize(size);
			textSub2 = this.niceSize(avail);
		} else {
			percentString = "";
		}
		if(size <= 0) {
			textSub1 = _("Not Found");
			textSub2 = "";
		}

		// set label contents
		let textpercent_y = Math.round((absoluteSize * global.ui_scale) / 2 - fontSize * (1.26 * global.ui_scale));
		this.textpercent.set_position(null, textpercent_y);
		this.textpercent.set_text(percentString);
		this.textpercent.style = "font-size: " + fontSize + "px;"
								+ "font-family: '" + this.font["font-family"] + "';"
								+ "font-style:" + this.font["font-style"] + ";"
								+ "font-weight:" + this.font["font-weight"] + ";"
								+ "font-stretch:" + this.font["font-stretch"] + ";"
								+ "width: " + absoluteSize + "px;"
								+ "color: " + this.text_color + ";";

		let textsub_y = Math.round(textpercent_y + fontSize * (1.25 * global.ui_scale));
		this.textsub.set_position(null, textsub_y);
		this.textsub.set_text(textSub1);
		this.textsub.style = "font-size: " + fontSizeSub + "px;"
							+ "font-family: '" + this.fontSub["font-family"] + "';"
							+ "font-style:" + this.fontSub["font-style"] + ";"
							+ "font-weight:" + this.fontSub["font-weight"] + ";"
							+ "font-stretch:" + this.fontSub["font-stretch"] + ";"
							+ "width: " + absoluteSize + "px;"
							+ "color: " + this.text_color + ";";

		let textsub2_y = Math.round(textsub_y + fontSizeSub * (1.25 * global.ui_scale));
		this.textsub2.set_position(null, textsub2_y);
		this.textsub2.set_text(textSub2);
		this.textsub2.style = "font-size: " + fontSizeSub + "px;"
							+ "font-family: '" + this.fontSub["font-family"] + "';"
							+ "font-style:" + this.fontSub["font-style"] + ";"
							+ "font-weight:" + this.fontSub["font-weight"] + ";"
							+ "font-stretch:" + this.fontSub["font-stretch"] + ";"
							+ "width: " + absoluteSize + "px;"
							+ "color: " + this.text_color + ";";

		//global.log("Redraw Done"); // debug
	},

	niceSize: function(value) {
		const prefixes = [" B", " K", " M", " G", " T"];
		
		let numerBase = 1000;
		if(this.size_prefix == "binary") numerBase = 1024;

		// First calculate how big the value to select a proper unit prefix. 
		let factor;
		if(value < numerBase) factor = 0;
		else if(value < numerBase ** 2) factor = 1;
		else if(value < numerBase ** 3) factor = 2;
		else if(value < numerBase ** 4) factor = 3;
		else factor = 4;

		let prefix = prefixes[factor];
		if(this.size_prefix == "binary") prefix += "i";

		// Scale value so that it matches a unit ex. 2000 -> 2K
		let scaledValue = value / (numerBase ** factor);

		return scaledValue.toFixed(this.size_fraction_digits) + prefix;

	},

	shortText: function(value, max = 9) {
		return (value.length > max) ? value.substr(0, max-1) + "…" : value;
	},


    /**
    * Parse raw font string.
    * @param {string} font_string - Font descriptor string
    * @returns {{"font-family": string, "font-size": Number, "font-weight": Number, "font-style": string, "font-stretch": string}} Font descriptor object
    */
    parseFontStringToCSS: function(font_string) {
        // Some fonts don't work, so a fallback font is a good idea
        const fallback_font_str = "Ubuntu Regular 16";
    
        // String are passed by reference here
        // make sure to copy the string to avoid triggering settings callback on change
        const font_string_copy = font_string.slice().trim();
        
        let css_font;
        try {
            const my_font_description = Pango.font_description_from_string(font_string_copy);
            css_font = this._PangoFontDescriptionToCSS(my_font_description);
        } catch (e) {
            Main.notifyError(
                _("Sorry, this font is not supported, please select a different one.") 
                + _(" Font: `") + font_string_copy + _("` Error: ") 
                + e.toString()
            );

            const fallback_font_description = Pango.font_description_from_string(fallback_font_str);
            css_font = this._PangoFontDescriptionToCSS(fallback_font_description);
        } finally {
            return css_font;
        }
        
    },


    /**
    * Process Pango.FontDescription and return valid CSS values
    * @param {Pango.FontDescription} font_description - Font descriptor
    * @returns {{"font-family": string, "font-size": Number, "font-weight": Number, "font-style": string, "font-stretch": string}} Font descriptor object
    */
    _PangoFontDescriptionToCSS: function(font_description) {
        const PangoStyle_to_CSS_map = {
            [Pango.Style.NORMAL]: "normal", 
            [Pango.Style.OBLIQUE]: "oblique", 
            [Pango.Style.ITALIC]: "italic", 
        };

        // font-stretch CSS property seems to be ignored by the CSS renderer
        const PangoStretch_to_CSS_map = {
            [Pango.Stretch.ULTRA_CONDENSED]: "ultra-condensed", 
            [Pango.Stretch.EXTRA_CONDENSED]: "extra-condensed", 
            [Pango.Stretch.CONDENSED]: "condensed", 
            [Pango.Stretch.NORMAL]: "normal", 
            [Pango.Stretch.SEMI_EXPANDED]: "semi-expanded", 
            [Pango.Stretch.EXPANDED]: "expanded", 
            [Pango.Stretch.EXTRA_EXPANDED]: "extra-expanded", 
            [Pango.Stretch.ULTRA_EXPANDED]: "ultra-expanded", 
        };
        
        return {
            "font-family": font_description.get_family(),
            "font-size": Math.floor(font_description.get_size() / Pango.SCALE),
            "font-weight": font_description.get_weight(),
            "font-style": PangoStyle_to_CSS_map[font_description.get_style()],
            "font-stretch": PangoStretch_to_CSS_map[font_description.get_stretch()]
        };
    },

	refreshDecoration: function() {
		// desklet label (header)
		let fs = decodeURIComponent(this.filesystem.replace("file://", "").trim());
		if(this.use_custom_label == true)
			this.setHeader(this.custom_label)
		else if(this.type == "ram")
			this.setHeader(_("RAM"));
		else if(this.type == "swap")
			this.setHeader(_("Swap"));
		else if(fs == null || fs == "" || fs == "/")
			this.setHeader(_("Disk Space"));
		else {
			let pathparts = fs.split("/");
			this.setHeader(pathparts[pathparts.length-1]);
		}

		// prevent decorations?
		this.metadata["prevent-decorations"] = this.hide_decorations;
		this._updateDecoration();
	},

	on_setting_changed: function() {
		// update decoration settings
		this.refreshDecoration();

		// settings changed; instant refresh
		Mainloop.source_remove(this.timeout);
		this.update();
	},

	on_desklet_clicked: function() {
		if(this.onclick_action == "filemanager") {
			let fs = decodeURIComponent(this.filesystem.replace("file://", "").trim());
			Util.spawnCommandLine("xdg-open " + '"' + fs + '"');
		} else if(this.onclick_action == "partitionmanager") {
				Util.spawnCommandLine("gnome-disks");
		} else if(this.onclick_action == "sysmonitor") {
				Util.spawnCommandLine("gnome-system-monitor");
		}
	},

	on_desklet_removed: function() {
		Mainloop.source_remove(this.timeout);
	}
}
